//+------------------------------------------------------------------+
//|                                          FX5_Divergence_V2.1.mq5 |
//|                                         forexsynthetics.com       |
//|                                   https://forexsynthetics.com     |
//+------------------------------------------------------------------+
#property copyright "forexsynthetics.com"
#property link      "https://forexsynthetics.com"
#property version   "2.10"
#property indicator_separate_window
#property indicator_buffers 5
#property indicator_plots   4

//---- plot upOsMA
#property indicator_label1  "Up OSMA"
#property indicator_type1   DRAW_HISTOGRAM
#property indicator_color1  clrLimeGreen
#property indicator_style1  STYLE_SOLID
#property indicator_width1  2

//---- plot downOsMA
#property indicator_label2  "Down OSMA"
#property indicator_type2   DRAW_HISTOGRAM
#property indicator_color2  clrFireBrick
#property indicator_style2  STYLE_SOLID
#property indicator_width2  2

//---- plot bullishDivergence
#property indicator_label3  "Bullish Divergence"
#property indicator_type3   DRAW_ARROW
#property indicator_color3  clrGreen
#property indicator_style3  STYLE_SOLID
#property indicator_width3  1

//---- plot bearishDivergence
#property indicator_label4  "Bearish Divergence"
#property indicator_type4   DRAW_ARROW
#property indicator_color4  clrRed
#property indicator_style4  STYLE_SOLID
#property indicator_width4  1

//---- input parameters
input group "*** OSMA Settings ***"
input int       fastEMA = 12;
input int       slowEMA = 26;
input int       signal = 9;
input ENUM_APPLIED_PRICE InpAppliedPrice = PRICE_CLOSE;

input group "*** Indicator Settings ***"
input bool      drawDivergenceLines = true;
input bool      displayAlert = true;

//---- buffers
double upOsMA[];
double downOsMA[];
double bullishDivergence[];
double bearishDivergence[];
double OsMA[];

//---- indicator handles
int osmaHandle;

//---- global variables
static datetime lastAlertTime;
string shortName;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
   //---- indicator buffers mapping
   SetIndexBuffer(0, upOsMA, INDICATOR_DATA);
   SetIndexBuffer(1, downOsMA, INDICATOR_DATA);
   SetIndexBuffer(2, bullishDivergence, INDICATOR_DATA);
   SetIndexBuffer(3, bearishDivergence, INDICATOR_DATA);
   SetIndexBuffer(4, OsMA, INDICATOR_CALCULATIONS);
   
   //---- setting arrow codes
   PlotIndexSetInteger(2, PLOT_ARROW, 233);
   PlotIndexSetInteger(3, PLOT_ARROW, 234);
   
   //---- setting empty values
   PlotIndexSetDouble(0, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(1, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(2, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(3, PLOT_EMPTY_VALUE, 0.0);
   
   //---- create OSMA handle
   osmaHandle = iOsMA(_Symbol, _Period, fastEMA, slowEMA, signal, InpAppliedPrice);
   if(osmaHandle == INVALID_HANDLE)
   {
      Print("Failed to create OSMA indicator handle");
      return(INIT_FAILED);
   }
   
   //---- indicator short name
   shortName = "FX5_Divergence_v2.1(" + IntegerToString(fastEMA) + "," + 
               IntegerToString(slowEMA) + "," + IntegerToString(signal) + ")";
   IndicatorSetString(INDICATOR_SHORTNAME, shortName);
   
   //---- digits
   IndicatorSetInteger(INDICATOR_DIGITS, _Digits + 2);
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   //---- release indicator handle
   if(osmaHandle != INVALID_HANDLE)
      IndicatorRelease(osmaHandle);
   
   //---- delete divergence lines
   int total = ObjectsTotal(0, 0, OBJ_TREND);
   for(int i = total - 1; i >= 0; i--)
   {
      string label = ObjectName(0, i, 0, OBJ_TREND);
      if(StringFind(label, "DivergenceLine") >= 0)
         ObjectDelete(0, label);
   }
   
   ChartRedraw();
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   if(rates_total < slowEMA + signal + 10)
      return(0);
   
   if(BarsCalculated(osmaHandle) < rates_total)
      return(0);
   
   //---- copy OSMA values first (not as series)
   if(CopyBuffer(osmaHandle, 0, 0, rates_total, OsMA) <= 0)
      return(0);
   
   //---- set arrays as series after copying
   ArraySetAsSeries(upOsMA, true);
   ArraySetAsSeries(downOsMA, true);
   ArraySetAsSeries(bullishDivergence, true);
   ArraySetAsSeries(bearishDivergence, true);
   ArraySetAsSeries(OsMA, true);
   ArraySetAsSeries(time, true);
   ArraySetAsSeries(high, true);
   ArraySetAsSeries(low, true);
   
   //---- initialize buffers on first run
   if(prev_calculated == 0)
   {
      ArrayInitialize(upOsMA, 0);
      ArrayInitialize(downOsMA, 0);
      ArrayInitialize(bullishDivergence, 0);
      ArrayInitialize(bearishDivergence, 0);
   }
   
   int limit;
   if(prev_calculated == 0)
      limit = rates_total - slowEMA - signal - 5;
   else
      limit = rates_total - prev_calculated + 1;
   
   //---- main calculation loop
   for(int i = limit; i >= 0; i--)
   {
      //---- clear old values
      bullishDivergence[i] = 0;
      bearishDivergence[i] = 0;
      
      CalculateOsMA(i);
      
      if(i >= 2)
      {
         CatchBullishDivergence(i, rates_total, time, low);
         CatchBearishDivergence(i, rates_total, time, high);
      }
   }
   
   return(rates_total);
}

//+------------------------------------------------------------------+
//| Calculate OSMA histogram colors                                  |
//+------------------------------------------------------------------+
void CalculateOsMA(int i)
{
   if(OsMA[i] > 0)
   {
      upOsMA[i] = OsMA[i];
      downOsMA[i] = 0;
   }
   else if(OsMA[i] < 0)
   {
      downOsMA[i] = OsMA[i];
      upOsMA[i] = 0;
   }
   else
   {
      upOsMA[i] = 0;
      downOsMA[i] = 0;
   }
}

//+------------------------------------------------------------------+
//| Catch bullish divergence                                         |
//+------------------------------------------------------------------+
void CatchBullishDivergence(int shift, int rates_total, const datetime &time[], const double &low[])
{
   if(!IsIndicatorTrough(shift, rates_total))
      return;
   
   int currentTrough = shift;
   int lastTrough = GetIndicatorLastTrough(shift, rates_total);
   
   if(lastTrough < 0)
      return;
   
   //---- Classical bullish divergence
   if(OsMA[currentTrough] > OsMA[lastTrough] && low[currentTrough] < low[lastTrough])
   {
      bullishDivergence[currentTrough] = OsMA[currentTrough];
      
      if(drawDivergenceLines)
      {
         DrawPriceTrendLine(time[currentTrough], time[lastTrough], low[currentTrough],
                           low[lastTrough], clrGreen, STYLE_SOLID);
         DrawIndicatorTrendLine(time[currentTrough], time[lastTrough], OsMA[currentTrough],
                               OsMA[lastTrough], clrGreen, STYLE_SOLID);
      }
      
      if(displayAlert)
         DisplayAlert("Classical bullish divergence on: ", currentTrough, time);
   }
   
   //---- Reverse bullish divergence
   if(OsMA[currentTrough] < OsMA[lastTrough] && low[currentTrough] > low[lastTrough])
   {
      bullishDivergence[currentTrough] = OsMA[currentTrough];
      
      if(drawDivergenceLines)
      {
         DrawPriceTrendLine(time[currentTrough], time[lastTrough], low[currentTrough],
                           low[lastTrough], clrGreen, STYLE_DOT);
         DrawIndicatorTrendLine(time[currentTrough], time[lastTrough], OsMA[currentTrough],
                               OsMA[lastTrough], clrGreen, STYLE_DOT);
      }
      
      if(displayAlert)
         DisplayAlert("Reverse bullish divergence on: ", currentTrough, time);
   }
}

//+------------------------------------------------------------------+
//| Catch bearish divergence                                         |
//+------------------------------------------------------------------+
void CatchBearishDivergence(int shift, int rates_total, const datetime &time[], const double &high[])
{
   if(!IsIndicatorPeak(shift, rates_total))
      return;
   
   int currentPeak = shift;
   int lastPeak = GetIndicatorLastPeak(shift, rates_total);
   
   if(lastPeak < 0)
      return;
   
   //---- Classical bearish divergence
   if(OsMA[currentPeak] < OsMA[lastPeak] && high[currentPeak] > high[lastPeak])
   {
      bearishDivergence[currentPeak] = OsMA[currentPeak];
      
      if(drawDivergenceLines)
      {
         DrawPriceTrendLine(time[currentPeak], time[lastPeak], high[currentPeak],
                           high[lastPeak], clrRed, STYLE_SOLID);
         DrawIndicatorTrendLine(time[currentPeak], time[lastPeak], OsMA[currentPeak],
                               OsMA[lastPeak], clrRed, STYLE_SOLID);
      }
      
      if(displayAlert)
         DisplayAlert("Classical bearish divergence on: ", currentPeak, time);
   }
   
   //---- Reverse bearish divergence
   if(OsMA[currentPeak] > OsMA[lastPeak] && high[currentPeak] < high[lastPeak])
   {
      bearishDivergence[currentPeak] = OsMA[currentPeak];
      
      if(drawDivergenceLines)
      {
         DrawPriceTrendLine(time[currentPeak], time[lastPeak], high[currentPeak],
                           high[lastPeak], clrRed, STYLE_DOT);
         DrawIndicatorTrendLine(time[currentPeak], time[lastPeak], OsMA[currentPeak],
                               OsMA[lastPeak], clrRed, STYLE_DOT);
      }
      
      if(displayAlert)
         DisplayAlert("Reverse bearish divergence on: ", currentPeak, time);
   }
}

//+------------------------------------------------------------------+
//| Check if current bar is indicator peak                           |
//+------------------------------------------------------------------+
bool IsIndicatorPeak(int shift, int rates_total)
{
   if(shift + 1 >= rates_total || shift < 1)
      return(false);
   
   if(OsMA[shift] > 0 && OsMA[shift] > OsMA[shift+1] && OsMA[shift] > OsMA[shift-1])
   {
      for(int i = shift + 1; i < rates_total && i < shift + 100; i++)
      {
         if(OsMA[i] < 0)
            return(true);
         if(OsMA[i] > OsMA[shift])
            return(false);
      }
   }
   return(false);
}

//+------------------------------------------------------------------+
//| Check if current bar is indicator trough                         |
//+------------------------------------------------------------------+
bool IsIndicatorTrough(int shift, int rates_total)
{
   if(shift + 1 >= rates_total || shift < 1)
      return(false);
   
   if(OsMA[shift] < 0 && OsMA[shift] < OsMA[shift+1] && OsMA[shift] < OsMA[shift-1])
   {
      for(int i = shift + 1; i < rates_total && i < shift + 100; i++)
      {
         if(OsMA[i] > 0)
            return(true);
         if(OsMA[i] < OsMA[shift])
            return(false);
      }
   }
   return(false);
}

//+------------------------------------------------------------------+
//| Get last indicator peak                                          |
//+------------------------------------------------------------------+
int GetIndicatorLastPeak(int shift, int rates_total)
{
   for(int i = shift + 5; i < rates_total - 2; i++)
   {
      if(i + 2 >= rates_total || i < 2)
         continue;
      
      if(OsMA[i] >= OsMA[i+1] && OsMA[i] > OsMA[i+2] &&
         OsMA[i] >= OsMA[i-1] && OsMA[i] > OsMA[i-2])
         return(i);
   }
   return(-1);
}

//+------------------------------------------------------------------+
//| Get last indicator trough                                        |
//+------------------------------------------------------------------+
int GetIndicatorLastTrough(int shift, int rates_total)
{
   for(int i = shift + 5; i < rates_total - 2; i++)
   {
      if(i + 2 >= rates_total || i < 2)
         continue;
      
      if(OsMA[i] <= OsMA[i+1] && OsMA[i] < OsMA[i+2] &&
         OsMA[i] <= OsMA[i-1] && OsMA[i] < OsMA[i-2])
         return(i);
   }
   return(-1);
}

//+------------------------------------------------------------------+
//| Display alert                                                    |
//+------------------------------------------------------------------+
void DisplayAlert(string message, int shift, const datetime &time[])
{
   if(shift <= 2 && time[shift] != lastAlertTime)
   {
      lastAlertTime = time[shift];
      Alert(message, _Symbol, " , ", EnumToString((ENUM_TIMEFRAMES)_Period));
   }
}

//+------------------------------------------------------------------+
//| Draw price trend line                                            |
//+------------------------------------------------------------------+
void DrawPriceTrendLine(datetime x1, datetime x2, double y1, double y2, 
                        color lineColor, ENUM_LINE_STYLE style)
{
   string label = "DivergenceLine2.1# " + IntegerToString(x1);
   ObjectDelete(0, label);
   ObjectCreate(0, label, OBJ_TREND, 0, x1, y1, x2, y2);
   ObjectSetInteger(0, label, OBJPROP_RAY_RIGHT, false);
   ObjectSetInteger(0, label, OBJPROP_COLOR, lineColor);
   ObjectSetInteger(0, label, OBJPROP_STYLE, style);
   ObjectSetInteger(0, label, OBJPROP_WIDTH, 1);
   ObjectSetInteger(0, label, OBJPROP_BACK, true);
   ObjectSetInteger(0, label, OBJPROP_SELECTABLE, false);
}

//+------------------------------------------------------------------+
//| Draw indicator trend line                                        |
//+------------------------------------------------------------------+
void DrawIndicatorTrendLine(datetime x1, datetime x2, double y1, double y2,
                            color lineColor, ENUM_LINE_STYLE style)
{
   int indicatorWindow = ChartWindowFind(0, shortName);
   if(indicatorWindow < 0)
      return;
   
   string label = "DivergenceLine2.1$# " + IntegerToString(x1);
   ObjectDelete(0, label);
   ObjectCreate(0, label, OBJ_TREND, indicatorWindow, x1, y1, x2, y2);
   ObjectSetInteger(0, label, OBJPROP_RAY_RIGHT, false);
   ObjectSetInteger(0, label, OBJPROP_COLOR, lineColor);
   ObjectSetInteger(0, label, OBJPROP_STYLE, style);
   ObjectSetInteger(0, label, OBJPROP_WIDTH, 1);
   ObjectSetInteger(0, label, OBJPROP_BACK, true);
   ObjectSetInteger(0, label, OBJPROP_SELECTABLE, false);
}
//+------------------------------------------------------------------+
